


// GENERAL STORE AVAILABILITY

// when a general store is destroyed, check if there are any general
// stores remaining (if not, restart the general store creation task)
function GeneralStoreUnemploy::onMessage(%vertex, %type, %param)
{
   // check if a general store is destroyed
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }
   
   // get all of the general stores on the map
   %list = tsCreateObjList();
   tsFindObjTypeOnMap("General", "bldg", %list);
   
   // check if at least one general store is created
   %count = %list.getSize();
   for (%index = 0; %index < %count; %index++)
   {
      // make sure this object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // make sure the building is in the production state
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() != $BuildingState::Production)
      {
         continue;
      }
      
      // if there is at least one employee, keep this active
      if (%building.getEmployeeCount() > 0)
      {
         // pretent that this vertex was not affected (needs to be immediately
         // resolved, so the vertex is always active to receive messages)
         %vertex.decCompleteCount();
         TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreUnemploy", $TSV_AVAIL);
         tsDestroyObjList(%list);
         return;
      }
   }
   tsDestroyObjList(%list);
   
   // if there are no general stores, the general store creation
   // vertex needs to be restarted (disable general store profit)
   TaskModifier.strResetVertex("GeneralStore", "GeneralStoreEmploy");
   TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreEmploy", $TSV_AVAIL);
}

// check the number of actively placed honey pots to limit the count to 3
function GeneralStoreHoneyCreate::onUpdate(%vertex)
{
   // if the honey building has not been unlocked, do nothing
   if (isObject(BadgeHoneypot2) == false)
   {
      return;
   }
   if (BadgeHoneypot2.isComplete() == false)
   {
      return;
   }
   
   // get all of the honey pots on the map
   %list = tsCreateObjList();
   tsFindObjTypeOnMap("Honeypot", "bldg", %list);
   
   // count how many honey pots are available
   %honeyCount = 0;
   %count = %list.getSize();
   for (%index = 0; %index < %count; %index++)
   {
      // make sure this object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // if the honey building is not being destroyed, it contributes to the active
      // number of buildings for the purpose of disabling/enabling the building
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() != $BuildingState::Destruction)
      {
         %honeyCount++;
      }
   }
   tsDestroyObjList(%list);
   
   // disable the honey building placement if at least 3 honey pots have been placed
   HoneypotButton.disabled = (%honeyCount >= 3);
}

// when honey is created, check if three have been made
function GeneralStoreHoneyCreate::onMessage(%vertex, %type, %param)
{
   // check if a honey pot is created
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }
   
   // get all of the honey pots on the map
   %list = tsCreateObjList();
   tsFindObjTypeOnMap("Honeypot", "bldg", %list);
   
   // count how many honey pots are available
   %honeyCount = 0;
   %count = %list.getSize();
   for (%index = 0; %index < %count; %index++)
   {
      // make sure this object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // if there is at least one producing general store left, then
      // keep the general store destruction vertex open
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() == $BuildingState::Production)
      {
         %honeyCount++;
      }
   }
   tsDestroyObjList(%list);
   
   // if not enough honey pots have been created, keep this vertex active
   if (%honeyCount < 3)
   {
      // pretent that this vertex was not affected (needs to be immediately
      // resolved, so the vertex is always active to receive messages)
      %vertex.decCompleteCount();
      TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreHoneyCreate", $TSV_AVAIL);
      return;
   }
   
   // enable the destroy honey building vertex
   TaskModifier.strResetVertex("GeneralStore", "GeneralStoreHoneyDestroy");
   TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreHoneyDestroy", $TSV_AVAIL);
   
   // enable the honey button
   tsEnableCommand(HoneyjarCmdData, true);
   
   // if this is the first resource for the general store, restart profit
   if (GeneralStoreProfit.isAvailable() == true &&
      GeneralStorePumpkin.isComplete() == false &&
      GeneralStoreSheep.isComplete() == false)
   {
      TaskModifier.strResetVertex("GeneralStore", "GeneralStoreRestart");
      TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreRestart", $TSV_AVAIL);
   }
}

// check if a honey pot is destroyed (make sure that there are enough
// honey pots remaining on the map for the bonus)
function GeneralStoreHoneyDestroy::onMessage(%vertex, %type, %param)
{
   // check if a honey pot is created
   Parent::onMessage(%vertex, %type, %param);
   if (%vertex.isMarkedComplete() == false)
   {
      return;
   }
   
   // get all of the honey pots on the map
   %list = tsCreateObjList();
   tsFindObjTypeOnMap("Honeypot", "bldg", %list);
   
   // count how many honey pots are available
   %honeyCount = 0;
   %count = %list.getSize();
   for (%index = 0; %index < %count; %index++)
   {
      // make sure this object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // if there is at least one producing general store left, then
      // keep the general store destruction vertex open
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() == $BuildingState::Production)
      {
         %honeyCount++;
      }
   }
   tsDestroyObjList(%list);
   
   // if enough honey pots are available, restore the destroy
   if (%honeyCount >= 3)
   {
      // pretent that this vertex was not affected (needs to be immediately
      // resolved, so the vertex is always active to receive messages)
      %vertex.decCompleteCount();
      TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreHoneyDestroy", $TSV_AVAIL);
      return;
   }
   
   // if there are not enough honey pots, the general store creation
   // vertex needs to be restarted (disable general store profit)
   TaskModifier.strResetVertex("GeneralStore", "GeneralStoreHoneyCreate");
   TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreHoneyCreate", $TSV_AVAIL);
   
   // disable the honey button
   tsEnableCommand(HoneyjarCmdData, false);
   
   // enable the honey building placement
   HoneypotButton.disabled = false;
}

// check the number of actively placed town halls to limit the count to 1
function GeneralStoreTownHallMonitor::onUpdate(%vertex)
{
   // check if there is at least one lumber mill (required for
   // the town hall)
   
   // get all of the lumber mills on the map
   %list = tsCreateObjList();
   tsFindObjTypeOnMap("Lumbermill", "bldg", %list);
   
   // check if a lumber mill is found
   %found = false;
   %count = %list.getSize();
   for (%index = 0; %index < %count; %index++)
   {
      // make sure this object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // if the lumber mill is not producing, do not count it
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() == $BuildingState::Production)
      {
         %found = true;
         break;
      }
   }
   tsDestroyObjList(%list);
   
   // if no lumber mills are found, there's nothing to do here
   if (%found == false)
   {
      TownhallButton.disabled = true;
      return;
   }
   
   // get all of the town halls on the map
   %list = tsCreateObjList();
   tsFindObjTypeOnMap("TownHall", "bldg", %list);
   
   // count how many town halls are available
   %honeyCount = 0;
   %count = %list.getSize();
   for (%index = 0; %index < %count; %index++)
   {
      // make sure this object is on the player's team
      %object = %list.getID(%index);
      if (%object.getTeam() != $OST_PLAYER)
      {
         continue;
      }
      
      // if the hallo building is not being destroyed, it contributes to the active
      // number of buildings for the purpose of disabling/enabling the building
      %building = slgQueryInterface(%object, $CID_BUILDING);
      if (%building.getState() != $BuildingState::Destruction)
      {
         TownhallButton.disabled = true;
         return;
      }
   }
   tsDestroyObjList(%list);
   
   // at this point, there are no town halls
   TownhallButton.disabled = false;
}

// handle resource production and reenable the profit timer
function GeneralStoreProfit::onExit(%vertex)
{
   // find out how much gold should be rewarded
   %gold = 0;
   if (GeneralStorePumpkin.isComplete() == true) %gold += 5;
   if (GeneralStoreSheep.isComplete() == true) %gold += 5;
   if (GeneralStoreHoneyCreate.isComplete() == true) %gold += 5;
   
   // if gold should be rewarded, reward it to the general stores
   if (%gold > 0)
   {
      // get the general stores
      %list = tsCreateObjList();
      tsFindObjTypeOnMap("General", "bldg", %list);
      
      // for each of the general stores that is employed
      %count = %list.getSize();
      for (%index = 0; %index < %count; %index++)
      {
         // if this general store is not on the player's team, skip it
         %object = %list.getID(%index);
         if (%object.getTeam() != $OST_PLAYER)
         {
            continue;
         }
         
         // if the building is not producing, skip it
         %building = slgqueryinterface(%object, $CID_BUILDING);
         if (%building.getState() != $BuildingState::Production)
         {
            continue;
         }
         
         // if the building has an employee, give it resources
         if (%building.getEmployeeCount() > 0)
         {
            // reward the general store that was found
            tsModifyResource($Resource::Gold, %gold, true);
            SendProductionToClient(%object, %gold @ " Gold");
         }
      }
      tsDestroyObjList(%list);
   }
   
   // reenable the general store profit timer
   TaskModifier.strResetVertex("GeneralStore", "GeneralStoreProfit");
   TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreProfit", $TSV_AVAIL);
}

// stop the profit task and mark this vertex complete
function GeneralStoreRestart::onEnter(%vertex)
{
   TaskModifier.strResetVertex("GeneralStore", "GeneralStoreProfit");
   TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreRestart", $TSV_CMPLETE);
}

// enough time has passed that the profit vertex will restart the timer if restarted
function GeneralStoreRestart::onExit(%vertex)
{
   // check if the general store profit timer should be restarted
   if (GeneralStoreEmploy.isComplete() == true)
   {
      TaskModifier.strMarkVertex("GeneralStore", "GeneralStoreProfit", $TSV_AVAIL);
   }
}
